/**
 *  Copyright 2002-2025 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of SeBIE.
 *
 *  SeBIE is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  SeBIE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SeBIE; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "sebie.h"
#include "sebie_helpers.h"
#include "sebie_view.h"
#include "sebie_gtk_helpers.h"
#include "sebie_config.h"
#include "sebie_callbacks.h"
#include "config.h"

#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#if GTK_CHECK_VERSION(4,0,0)
void
sebie_update_view(struct sebie_s *sebie,
                  GtkDrawingArea *area,
                  cairo_t *cr)
#else
void
sebie_update_view(struct sebie_s *sebie)
#endif
{
  if (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_FAC) {
    gchar buf[6];
    sebie->model.selection_fac = (double)sebie->model.selection_fac_width / (double) sebie->model.selection_fac_height;
    snprintf(buf, 6, "%5.2f", sebie->model.selection_fac);
    gtk_label_set_text(GTK_LABEL(sebie->view.label_selection_fac), buf);
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_FAC_W) {
    gchar buf[6];
    snprintf(buf, 6, "%d", sebie->model.selection_fac_width);
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_selection_fac_width));
    gtk_entry_buffer_set_text(entry_buffer, buf, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_selection_fac_width), buf);
#endif
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_FAC_H) {
    gchar buf[6];
    snprintf(buf, 6, "%d", sebie->model.selection_fac_height);
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_selection_fac_height));
    gtk_entry_buffer_set_text(entry_buffer, buf, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_selection_fac_height), buf);
#endif
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_IMAGE_GAMMA) {
    gtk_spin_button_set_value (GTK_SPIN_BUTTON(sebie->view.entry_selection_gamma), sebie->model.image_gamma);
  }

#if GTK_CHECK_VERSION(4,0,0)
  if (sebie->dirty_tags & SEBIE_UPDATE_SCALED_PIXBUF ||
      sebie->dirty_tags & SEBIE_UPDATE_SELECTION_RECT) {
      gtk_widget_queue_draw(sebie->view.drawingarea1);
  }
#else
  if (sebie->dirty_tags & SEBIE_UPDATE_SCALED_PIXBUF ||
      sebie->dirty_tags & SEBIE_UPDATE_SELECTION_RECT) {
#endif
    /* do it the rough way, always draw the whole area and not just the
     * areas which changed  */
#if GTK_CHECK_VERSION(4,0,0)
  if (area != NULL && cr != NULL) {
#elif GTK_CHECK_VERSION(3,0,0)
#if GTK_CHECK_VERSION(3,22,0)
    GdkWindow *window = gtk_widget_get_window(sebie->view.drawingarea1);
    cairo_region_t *cairo_region = cairo_region_create();
    GdkDrawingContext *drawing_context = gdk_window_begin_draw_frame (window, cairo_region);
    cairo_t *cr = gdk_drawing_context_get_cairo_context(drawing_context);
#else
    GdkWindow *window = gtk_widget_get_window(sebie->view.drawingarea1);
    cairo_t *cr = gdk_cairo_create (window);
#endif
#else
    GdkGC *gdkGC = gdk_gc_new(sebie->view.drawingarea1->window);
#endif

    // SEBIE_UPDATE_SELECTION_RECT could be optimized to (SEBIE_UPDATE_SELECTION_RECT && rect extens scaled_pixbuf area)
    if ((sebie->dirty_tags & SEBIE_UPDATE_SCALED_PIXBUF) || (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_RECT)) {
#if GTK_CHECK_VERSION(3,0,0)
#if GTK_CHECK_VERSION(4,12,0)
      int drawingarea1_width = gtk_widget_get_width(sebie->view.drawingarea1);
      int drawingarea2_height = gtk_widget_get_height(sebie->view.drawingarea1);
#else
      int drawingarea1_width = gtk_widget_get_allocated_width(sebie->view.drawingarea1);
      int drawingarea2_height = gtk_widget_get_allocated_height(sebie->view.drawingarea1);
#endif
      cairo_set_source_rgb(cr, 0.878, 0.878, 0.878);
      cairo_rectangle (cr, 0, 0, drawingarea1_width, drawingarea2_height);
      cairo_fill(cr);
#else
      GdkColor color;
      color.red   = 0xe0e0;
      color.green = 0xe0e0;
      color.blue  = 0xe0e0;
      gdk_gc_set_rgb_fg_color(gdkGC, &color);
      gdk_draw_rectangle(sebie->view.drawingarea1->window,
                         gdkGC,
                         TRUE,
                         0, 0,
                         -1, -1);
#endif
    }

    if (sebie->model.scaled_pixbuf != NULL) {
#if GTK_CHECK_VERSION(3,0,0)
      gdk_cairo_set_source_pixbuf (cr, sebie->model.scaled_pixbuf, 0, 0);
      cairo_paint (cr);
#else
      gdk_pixbuf_render_to_drawable (sebie->model.scaled_pixbuf,
                                     sebie->view.drawingarea1->window,
                                     gdkGC,
                                     0, 0,
                                     0, 0,
                                     sebie->model.scaled_width,
                                     sebie->model.scaled_height,
                                     GDK_RGB_DITHER_NONE,
                                     0,
                                     0);
#endif
    }

    gint rx, ry, rheight, rwidth;

    if (sebie->model.selection_width > 0) {
      rx = sebie->model.selection_x;
      rwidth = sebie->model.selection_width;
    } else {
      rx = sebie->model.selection_x + sebie->model.selection_width;
      rwidth = - sebie->model.selection_width;
    }

    if (sebie->model.selection_height > 0) {
      ry = sebie->model.selection_y;
      rheight = sebie->model.selection_height;
    } else {
      ry = sebie->model.selection_y + sebie->model.selection_height;
      rheight = - sebie->model.selection_height;
    }
#if GTK_CHECK_VERSION(3,0,0)
    cairo_set_source_rgb (cr, 1., 1., 1.);
    cairo_set_operator (cr, CAIRO_OPERATOR_DIFFERENCE);
    cairo_rectangle (cr, rx, ry, rwidth, rheight);
    cairo_set_line_width(cr, 1);
    cairo_stroke(cr);
#else
    gdk_gc_set_function(gdkGC, GDK_INVERT);
    gdk_draw_rectangle (sebie->view.drawingarea1->window,
                        gdkGC,
                        FALSE,
                        rx,
                        ry,
                        rwidth,
                        rheight);
#endif

#if GTK_CHECK_VERSION(4,0,0)
#elif GTK_CHECK_VERSION(3,0,0)
#if GTK_CHECK_VERSION(3,22,0)
    gdk_window_end_draw_frame(window,drawing_context);
    cairo_region_destroy(cairo_region);
#else
    cairo_paint (cr);
    cairo_destroy (cr);
#endif
#endif
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_INPUT_PATH) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_input_path));
    gtk_entry_buffer_set_text(entry_buffer, sebie->model.input_path, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_input_path),
                       sebie->model.input_path);
#endif
  }
  if (sebie->dirty_tags & SEBIE_UPDATE_INPUT_REGEX) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_input_regex));
    gtk_entry_buffer_set_text(entry_buffer, sebie->model.input_regex, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_input_regex),
                       sebie->model.input_regex);
#endif
  }
  if (sebie->dirty_tags & SEBIE_UPDATE_OUTPUT_FORMAT) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_output_format));
    gtk_entry_buffer_set_text(entry_buffer, sebie->model.output_format, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_output_format),
                       sebie->model.output_format);
#endif
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_OUTPUT_FILE_NAME) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(sebie->view.entry_output_file));
    gtk_entry_buffer_set_text(entry_buffer, sebie->model.output_file_name, -1);
#else
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_output_file),
                       sebie->model.output_file_name);
#endif
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_STATUS_BAR) {
#if GTK_CHECK_VERSION(4,10,0)
    gtk_label_set_text(GTK_LABEL(sebie->view.statusbar1), sebie->model.statusbar_text);
#else
    gtk_statusbar_pop(GTK_STATUSBAR(sebie->view.statusbar1), sebie->view.statusbar1_context_id);
    gtk_statusbar_push(GTK_STATUSBAR(sebie->view.statusbar1), sebie->view.statusbar1_context_id, sebie->model.statusbar_text);
#endif
  }

  sebie->dirty_tags = 0;
  save_sebie_config(sebie);
}


void
sebie_error_popup(struct sebie_s *sebie, const gchar *error_str)
{
#if GTK_CHECK_VERSION(4,10,0)
  GtkAlertDialog *dialog = gtk_alert_dialog_new ("%s", error_str);
  gtk_alert_dialog_show (dialog, GTK_WINDOW(sebie->view.window1));
  g_object_unref (dialog);
#else
  GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW(sebie->view.window1),
                                              GTK_DIALOG_DESTROY_WITH_PARENT,
                                              GTK_MESSAGE_ERROR,
                                              GTK_BUTTONS_CLOSE,
                                              "%s",
                                              error_str);

  // modal dialog: you must click to resume
#if GTK_CHECK_VERSION(4,0,0)
  gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);
  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK (on_error_dialog_response),
                    NULL);
  gtk_widget_show (dialog);
#else
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
#endif
#endif
}


static void
sebie_file_list_clear(struct sebie_s *sebie)
{
#if GTK_CHECK_VERSION(4,10,0)
    guint n = g_list_model_get_n_items (G_LIST_MODEL(sebie->model.file_list));
    for(; n > 0; n--) {
        gtk_string_list_remove (sebie->model.file_list, n - 1);
    }
#else
  if (sebie->model.file_list == NULL) {
    return;
  }
  gtk_list_store_clear(sebie->model.file_list);
#endif
}


static void
sebie_file_list_add(struct sebie_s *sebie, const gchar *filename)
{
#if GTK_CHECK_VERSION(4,10,0)
    gtk_string_list_append(sebie->model.file_list, filename);
#else
  if (sebie->model.file_list == NULL) {
    sebie->model.file_list = gtk_list_store_new(1, G_TYPE_STRING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(sebie->view.treeview_file_list), GTK_TREE_MODEL(sebie->model.file_list));
  }
  GtkTreeIter iter;
  gtk_list_store_append(sebie->model.file_list, &iter);
  gtk_list_store_set(sebie->model.file_list, &iter, 0, filename, -1);
#endif
}

void
load_pixbuf (struct sebie_s *sebie)
{
  if (sebie->model.input_file_name == NULL) {
    return;
  }

  GError *error = NULL;
  GdkPixbuf *pixbuf;

  gchar pathname[PATH_MAX];
  g_snprintf(pathname, PATH_MAX, "%s/%s", sebie->model.input_path, sebie->model.input_file_name);
  gchar statusbar_text[PATH_MAX + 64];
  g_snprintf(statusbar_text, PATH_MAX + 64, "   Load file: '%s'", pathname);
  sebie_set_string(&sebie->model.statusbar_text, statusbar_text);
  sebie->dirty_tags |= SEBIE_UPDATE_STATUS_BAR;

  if ((pixbuf = gdk_pixbuf_new_from_file(pathname, &error)) == NULL) {
    g_print("gdk_pixbuf_new_from_file failed (%s).\n", error->message);
    return;
  }
  sebie->model.image_width = gdk_pixbuf_get_width(pixbuf);
  sebie->model.image_height = gdk_pixbuf_get_height(pixbuf);
  if (sebie->model.image_pixbuf != NULL) {
    g_object_unref(sebie->model.image_pixbuf);
  }
  sebie->model.image_pixbuf = pixbuf;
  double scale1 = (double)sebie->model.pixbuf_width / (double)sebie->model.image_width;
  double scale2 = (double)sebie->model.pixbuf_height / (double)sebie->model.image_width;
  double scale3 = (double)sebie->model.pixbuf_width / (double)sebie->model.image_height;
  double scale4 = (double)sebie->model.pixbuf_height / (double)sebie->model.image_height;

  scale1 = MIN(scale1, scale2);
  scale1 = MIN(scale1, scale3);
  scale1 = MIN(scale1, scale4);
  sebie_debug("scale: %f \n", scale1);
  sebie->model.scaled_scale = scale1;
  sebie->model.scaled_width  = (int)((double)sebie->model.image_width * scale1);
  sebie->model.scaled_height = (int)((double)sebie->model.image_height * scale1);
  sebie->model.scaled_rotation = SEBIE_ROTATE_000;

  sebie_debug("scaled: width: %d height: %d \n",
              sebie->model.scaled_width,
              sebie->model.scaled_height);
  if (sebie->model.scaled_pixbuf_orig != NULL) {
    g_object_unref(sebie->model.scaled_pixbuf_orig);
  }
  if (sebie->model.scaled_pixbuf != NULL) {
    g_object_unref(sebie->model.scaled_pixbuf);
  }
  sebie->model.scaled_pixbuf_orig = gdk_pixbuf_scale_simple(sebie->model.image_pixbuf,
                                                            sebie->model.scaled_width,
                                                            sebie->model.scaled_height,
                                                            GDK_INTERP_HYPER);
  sebie->model.scaled_pixbuf = pixbuf_color_corrections(sebie->model.scaled_pixbuf_orig,
                                                        sebie->model.image_gamma,
                                                        sebie->model.image_grayscale);
  sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
#if 0
  sebie_update_view(sebie);
#endif
}


void
sebie_update_file_list(struct sebie_s *sebie)
{
  GDir *gdir;
  save_sebie_config(sebie);
  sebie_file_list_clear(sebie);
  if ((gdir = g_dir_open(sebie->model.input_path, 0, NULL)) == NULL) {
    return;
  }

  const gchar *next;
  while((next = g_dir_read_name(gdir)) != NULL) {
    sebie_debug("next %s\n", next);

    if (sebie->model.regex_status) {
      if (regexec(&sebie->model.regex, next, 0, NULL, 0) == 0) {
          sebie_file_list_add(sebie, next);
      }
    } else {
      sebie_debug("sebie->model.regex_status invalid");
    }
  }

  g_dir_close(gdir);
}


void
rotate_scaled(struct sebie_s *sebie, sebie_rotation direction)
{
  if (direction == SEBIE_ROTATE_090 ||
      direction == SEBIE_ROTATE_270 ) {
    GdkPixbuf *rotated = rotate_pixbuf(sebie->model.scaled_pixbuf, direction);

    sebie->model.scaled_width = gdk_pixbuf_get_width(rotated);
    sebie->model.scaled_height = gdk_pixbuf_get_height(rotated);
    if (sebie->model.scaled_pixbuf != NULL) {
      g_object_unref(sebie->model.scaled_pixbuf);
    }
    sebie->model.scaled_pixbuf = rotated;

    sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
#if GTK_CHECK_VERSION(4,0,0)
    sebie_update_view(sebie, NULL, NULL);
#else
    sebie_update_view(sebie);
#endif
  }
}


void
apply_color_corrections(struct sebie_s *sebie)
{
  if (sebie->model.scaled_pixbuf_orig != NULL &&
      sebie->model.scaled_pixbuf != NULL) {
    GdkPixbuf *corr = pixbuf_color_corrections(sebie->model.scaled_pixbuf_orig,
                                               sebie->model.image_gamma,
                                               sebie->model.image_grayscale);
    g_object_unref(sebie->model.scaled_pixbuf);

    if (sebie->model.scaled_rotation == SEBIE_ROTATE_000) {
      sebie->model.scaled_pixbuf = corr;
    } else {
      GdkPixbuf *rotated = rotate_pixbuf(corr, sebie->model.scaled_rotation);
      if (corr != NULL) {
        g_object_unref(corr);
      }
      sebie->model.scaled_pixbuf = rotated;
    }

    sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
  }
#if GTK_CHECK_VERSION(4,0,0)
  sebie_update_view(sebie, NULL, NULL);
#else
  sebie_update_view(sebie);
#endif
}


void
save_sebie_config(struct sebie_s *sebie)
{
  if (sebie->model_tags) {
    sebie_config_spec config_spec[] = {
      { .name = "InputPath", .type = CONFIG_STRING, .storage.string = &sebie->model.input_path, .defvalue.string = "" },
      { .name = "InputRegEx", .type = CONFIG_STRING, .storage.string = &sebie->model.input_regex, .defvalue.string = "\\([0-9]*\\).jpg" },
      { .name = "OutputFormat", .type = CONFIG_STRING, .storage.string = &sebie->model.output_format, .defvalue.string = "Image%s.jpg" },
      { .name = "ImageOutputWidth", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.selection_fac_width, .defvalue.integer = 1024},
      { .name = "ImageOutputHeight", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.selection_fac_height, .defvalue.integer = 768},
      { .name = "ImageGamma", .type = CONFIG_DOUBLE, .storage.doublev = &sebie->model.image_gamma, .defvalue.doublev = 1.0},
      { .name = "ImageGrayscale", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.image_grayscale, .defvalue.integer = 0},
      { .name = "Window1Width", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.window1_width, .defvalue.integer = 1187},
      { .name = "Window1Height", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.window1_height, .defvalue.integer = 832},
      { .name = "PixbufWidth", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.pixbuf_width, .defvalue.integer = 1024},
      { .name = "PixbufHeight", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.pixbuf_height, .defvalue.integer = 768},
      { .name = NULL, .type = CONFIG_END, .storage.string = NULL, .defvalue.string = NULL}
    };

    sebie_write_config(sebie->config_file, config_spec);

    sebie->model_tags = 0;
  }
}

#define SEBIE_N_MATCH  2

void
sebie_set_selected_image(struct sebie_s *sebie, const char *filename)
{
  sebie_set_string(&sebie->model.input_file_name, filename);
  load_pixbuf(sebie);
  if (sebie->model.regex_status) {
    regmatch_t matchptr[SEBIE_N_MATCH];
    if (regexec(&sebie->model.regex, sebie->model.input_file_name, SEBIE_N_MATCH, matchptr, 0) == 0) {
      char buf_regex[SEBIE_N_MATCH][256];
      int i;
      for (i = 0; i < SEBIE_N_MATCH; i++) {
        strncpy(buf_regex[i],sebie->model.input_file_name +  matchptr[i].rm_so, matchptr[i].rm_eo - matchptr[i].rm_so);
        buf_regex[i][matchptr[i].rm_eo - matchptr[i].rm_so] = '\0';
        sebie_debug("buf_regex[%d] = %s\n", i, buf_regex[i]);
      }
      char buf[256];
      snprintf(buf, 256, sebie->model.output_format, buf_regex[1]);
      sebie_set_string(&sebie->model.output_file_name, buf);
      sebie->dirty_tags |= SEBIE_UPDATE_OUTPUT_FILE_NAME;
#if GTK_CHECK_VERSION(4,0,0)
      sebie_update_view(sebie, NULL, NULL);
#else
      sebie_update_view(sebie);
#endif
    }
  } else {
      sebie_debug("sebie->model.regex_status invalid");
  }
}

#undef SEBIE_N_MATCH

static void
init_sebie_model(struct sebie_s *sebie)
{
  char *homedir;
  char buf[PATH_MAX];

  if ((homedir = getenv("HOME")) != NULL) {
    snprintf(buf, PATH_MAX, "%s/%s", homedir, ".sebieconf");
    sebie->config_file = g_strdup(buf);
  } else {
    sebie->config_file = g_strdup(".sebieconf");
  }

  sebie_config_spec config_spec[] = {
    { .name = "InputPath", .type = CONFIG_STRING, .storage.string = &sebie->model.input_path, .defvalue.string = "" },
    { .name = "InputRegEx", .type = CONFIG_STRING, .storage.string = &sebie->model.input_regex, .defvalue.string = "\\([0-9]*\\).jpg" },
    { .name = "OutputFormat", .type = CONFIG_STRING, .storage.string = &sebie->model.output_format, .defvalue.string = "Image%s.jpg" },
    { .name = "ImageOutputWidth", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.selection_fac_width, .defvalue.integer = 1024},
    { .name = "ImageOutputHeight", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.selection_fac_height, .defvalue.integer = 768},
    { .name = "ImageGamma", .type = CONFIG_DOUBLE, .storage.doublev = &sebie->model.image_gamma, .defvalue.doublev = 1.0},
    { .name = "ImageGrayscale", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.image_grayscale, .defvalue.integer = 0},
    { .name = "Window1Width", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.window1_width, .defvalue.integer = 1187},
    { .name = "Window1Height", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.window1_height, .defvalue.integer = 832},
    { .name = "PixbufWidth", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.pixbuf_width, .defvalue.integer = 1024},
    { .name = "PixbufHeight", .type = CONFIG_INTEGER, .storage.integer = &sebie->model.pixbuf_height, .defvalue.integer = 768},
    { .name = NULL, .type = CONFIG_END, .storage.string = NULL, .defvalue.string = NULL}
  };

  sebie_read_config(sebie->config_file, config_spec);
  if (sebie->model.selection_fac_width < 1) {

      sebie->model.selection_fac_width = 1;
  }
  if (sebie->model.selection_fac_height < 1) {
      sebie->model.selection_fac_height = 1;
  }
  sebie->model.regex_status = 0;

  sebie->dirty_tags |= SEBIE_UPDATE_INPUT_PATH;
  sebie->dirty_tags |= SEBIE_UPDATE_INPUT_REGEX;
  sebie->dirty_tags |= SEBIE_UPDATE_OUTPUT_FORMAT;

#if GTK_CHECK_VERSION(4,10,0)
  sebie->model.file_list = gtk_string_list_new (NULL);
  sebie->model.file_list_selection = gtk_single_selection_new (G_LIST_MODEL (sebie->model.file_list));
  gtk_single_selection_set_autoselect(sebie->model.file_list_selection, FALSE);
#else
  sebie->model.file_list = gtk_list_store_new(1, G_TYPE_STRING);
#endif

  sebie->model.selection_fac = sebie->model.selection_fac_width / sebie->model.selection_fac_height;

  sebie->model.mouse_click_type = SEBIE_NOT_CLICKED;

  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_W;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_H;
  sebie->dirty_tags |= SEBIE_UPDATE_IMAGE_GAMMA;

  sebie->model.statusbar_text = strdup("   SeBIE-" PACKAGE_VERSION " (c) 2002-2025 by Peter.Seiderer@ciselant.de");
  sebie->dirty_tags |= SEBIE_UPDATE_STATUS_BAR;

  sebie_regcomp(sebie);
}

static void
deinit_sebie_model(struct sebie_s *sebie)
{
    if (sebie->model.input_path != NULL) {
        g_free(sebie->model.input_path);
    }
    if (sebie->model.input_regex != NULL) {
        g_free(sebie->model.input_regex);
    }
    if (sebie->model.output_format != NULL) {
        g_free(sebie->model.output_format);
    }
    if (sebie->model.input_file_name != NULL) {
        g_free(sebie->model.input_file_name);
    }
    if (sebie->model.output_file_name != NULL) {
        g_free(sebie->model.output_file_name);
    }

    if (sebie->model.regex_status) {
        regfree(&sebie->model.regex);
        sebie->model.regex_status = 0;
    }

    if (sebie->model.image_pixbuf != NULL) {
        g_object_unref(sebie->model.image_pixbuf);
    }

    if (sebie->model.scaled_pixbuf_orig != NULL) {
        g_object_unref(sebie->model.scaled_pixbuf_orig);
    }

    if (sebie->model.scaled_pixbuf != NULL) {
        g_object_unref(sebie->model.scaled_pixbuf);
    }

    g_free(sebie->model.statusbar_text);

    g_free(sebie->config_file);
}

int main(G_GNUC_UNUSED int argc, G_GNUC_UNUSED char *argv[])
{
  GtkWidget *window1;
  struct sebie_s *sebie;

  sebie = g_new0(struct sebie_s, 1);

#if GTK_CHECK_VERSION(3,0,0)
#else
  gtk_set_locale ();
#endif

#if GTK_CHECK_VERSION(4,0,0)
  gtk_init ();
#else
  gtk_init (&argc, &argv);
#endif

  init_sebie_model(sebie);
  window1 = init_sebie_view(sebie);

#if GTK_CHECK_VERSION(4,0,0)
  sebie_update_view(sebie, NULL, NULL);
#else
  sebie_update_view(sebie);
#endif

#if GTK_CHECK_VERSION(4,10,0)
  gtk_widget_set_visible (window1, TRUE);
#else
  gtk_widget_show (window1);
#endif

#if GTK_CHECK_VERSION(4,0,0)
  while (g_list_model_get_n_items (gtk_window_get_toplevels ()) > 0) {
    g_main_context_iteration (NULL, TRUE);
  }
#else
  gtk_main ();
#endif

  deinit_sebie_model(sebie);
  g_free(sebie);

  exit(0);
}
