/**
 *  Copyright 2002 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of Jardin.
 *
 *  Jardin is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Jardin is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Foobar; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */
#include "jardin_jcvm.h"
#include "jardin_cap.h"
#include "jardin_debug.h"

#include <stdlib.h>


static char *jardin_jcvm_opcodestr[JARDIN_MAX_OPCODE];

static void jardin_jcvm_init_opcodestr() {
  jardin_jcvm_opcodestr[0x00] = "nop";
  jardin_jcvm_opcodestr[0x01] = "aconst_null";
  jardin_jcvm_opcodestr[0x02] = "sconst_m1";
  jardin_jcvm_opcodestr[0x03] = "sconst_0";
  jardin_jcvm_opcodestr[0x04] = "sconst_1";
  jardin_jcvm_opcodestr[0x05] = "sconst_2";
  jardin_jcvm_opcodestr[0x06] = "sconst_3";
  jardin_jcvm_opcodestr[0x07] = "sconst_4";
  jardin_jcvm_opcodestr[0x08] = "sconst_5";
  jardin_jcvm_opcodestr[0x09] = "iconst_m1";
  jardin_jcvm_opcodestr[0x0a] = "iconst_0";
  jardin_jcvm_opcodestr[0x0b] = "iconst_1";
  jardin_jcvm_opcodestr[0x0c] = "iconst_2";
  jardin_jcvm_opcodestr[0x0d] = "iconst_3";
  jardin_jcvm_opcodestr[0x0e] = "iconst_4";
  jardin_jcvm_opcodestr[0x0f] = "iconst_5";
  jardin_jcvm_opcodestr[0x10] = "bspush";
  jardin_jcvm_opcodestr[0x11] = "sspush";
  jardin_jcvm_opcodestr[0x12] = "bipush";
  jardin_jcvm_opcodestr[0x13] = "sipush";
  jardin_jcvm_opcodestr[0x14] = "iipush";
  jardin_jcvm_opcodestr[0x15] = "aload";
  jardin_jcvm_opcodestr[0x16] = "sload";
  jardin_jcvm_opcodestr[0x17] = "iload";
  jardin_jcvm_opcodestr[0x18] = "aload_0";
  jardin_jcvm_opcodestr[0x19] = "aload_1";
  jardin_jcvm_opcodestr[0x1a] = "aload_2";
  jardin_jcvm_opcodestr[0x1b] = "aload_3";
  jardin_jcvm_opcodestr[0x1c] = "sload_0";
  jardin_jcvm_opcodestr[0x1d] = "sload_1";
  jardin_jcvm_opcodestr[0x1e] = "sload_2";
  jardin_jcvm_opcodestr[0x1f] = "sload_3";
  jardin_jcvm_opcodestr[0x20] = "iload_0";
  jardin_jcvm_opcodestr[0x21] = "iload_1";
  jardin_jcvm_opcodestr[0x22] = "iload_2";
  jardin_jcvm_opcodestr[0x23] = "iload_3";
  jardin_jcvm_opcodestr[0x24] = "aaload";
  jardin_jcvm_opcodestr[0x25] = "baload";
  jardin_jcvm_opcodestr[0x26] = "saload";
  jardin_jcvm_opcodestr[0x27] = "iaload";
  jardin_jcvm_opcodestr[0x28] = "astore";
  jardin_jcvm_opcodestr[0x29] = "sstore";
  jardin_jcvm_opcodestr[0x2a] = "istore";
  jardin_jcvm_opcodestr[0x2b] = "astore_0";
  jardin_jcvm_opcodestr[0x2c] = "astore_1";
  jardin_jcvm_opcodestr[0x2d] = "astore_2";
  jardin_jcvm_opcodestr[0x2e] = "astore_3";
  jardin_jcvm_opcodestr[0x2f] = "sstore_0";
  jardin_jcvm_opcodestr[0x30] = "sstore_1";
  jardin_jcvm_opcodestr[0x31] = "sstore_2";
  jardin_jcvm_opcodestr[0x32] = "sstore_3";
  jardin_jcvm_opcodestr[0x33] = "istore_0";
  jardin_jcvm_opcodestr[0x34] = "istore_1";
  jardin_jcvm_opcodestr[0x35] = "istore_2";
  jardin_jcvm_opcodestr[0x36] = "istore_3";
  jardin_jcvm_opcodestr[0x37] = "aastore";
  jardin_jcvm_opcodestr[0x38] = "bastore";
  jardin_jcvm_opcodestr[0x39] = "sastore";
  jardin_jcvm_opcodestr[0x3a] = "iastore";
  jardin_jcvm_opcodestr[0x3b] = "pop";
  jardin_jcvm_opcodestr[0x3c] = "pop2";
  jardin_jcvm_opcodestr[0x3d] = "dup";
  jardin_jcvm_opcodestr[0x3e] = "dup2";
  jardin_jcvm_opcodestr[0x3f] = "dup_x";
  jardin_jcvm_opcodestr[0x40] = "swap_x";
  jardin_jcvm_opcodestr[0x41] = "sadd";
  jardin_jcvm_opcodestr[0x42] = "iadd";
  jardin_jcvm_opcodestr[0x43] = "ssub";
  jardin_jcvm_opcodestr[0x44] = "isub";
  jardin_jcvm_opcodestr[0x45] = "smul";
  jardin_jcvm_opcodestr[0x46] = "imul";
  jardin_jcvm_opcodestr[0x47] = "sdiv";
  jardin_jcvm_opcodestr[0x48] = "idiv";
  jardin_jcvm_opcodestr[0x49] = "srem";
  jardin_jcvm_opcodestr[0x4a] = "irem";
  jardin_jcvm_opcodestr[0x4b] = "sneg";
  jardin_jcvm_opcodestr[0x4c] = "ineg";
  jardin_jcvm_opcodestr[0x4d] = "sshl";
  jardin_jcvm_opcodestr[0x4e] = "ishl";
  jardin_jcvm_opcodestr[0x4f] = "sshr";
  jardin_jcvm_opcodestr[0x50] = "ishr";
  jardin_jcvm_opcodestr[0x51] = "sushr";
  jardin_jcvm_opcodestr[0x52] = "iushr";
  jardin_jcvm_opcodestr[0x53] = "sand";
  jardin_jcvm_opcodestr[0x54] = "iand";
  jardin_jcvm_opcodestr[0x55] = "sor";
  jardin_jcvm_opcodestr[0x56] = "ior";
  jardin_jcvm_opcodestr[0x57] = "sxor";
  jardin_jcvm_opcodestr[0x58] = "ixor";
  jardin_jcvm_opcodestr[0x59] = "sinc";
  jardin_jcvm_opcodestr[0x5a] = "iinc";
  jardin_jcvm_opcodestr[0x5b] = "s2b";
  jardin_jcvm_opcodestr[0x5c] = "s2i";
  jardin_jcvm_opcodestr[0x5d] = "i2b";
  jardin_jcvm_opcodestr[0x5e] = "i2s";
  jardin_jcvm_opcodestr[0x5f] = "icmp";
  jardin_jcvm_opcodestr[0x60] = "ifeq";
  jardin_jcvm_opcodestr[0x61] = "ifne";
  jardin_jcvm_opcodestr[0x62] = "iflt";
  jardin_jcvm_opcodestr[0x63] = "ifge";
  jardin_jcvm_opcodestr[0x64] = "ifgt";
  jardin_jcvm_opcodestr[0x65] = "ifle";
  jardin_jcvm_opcodestr[0x66] = "ifnull";
  jardin_jcvm_opcodestr[0x67] = "ifnonull";
  jardin_jcvm_opcodestr[0x68] = "if_acmpeq";
  jardin_jcvm_opcodestr[0x69] = "if_acmpne";
  jardin_jcvm_opcodestr[0x6a] = "if_scmpeq";
  jardin_jcvm_opcodestr[0x6b] = "if_scmpne";
  jardin_jcvm_opcodestr[0x6c] = "if_scmplt";
  jardin_jcvm_opcodestr[0x6d] = "if_scmpge";
  jardin_jcvm_opcodestr[0x6e] = "if_scmpgt";
  jardin_jcvm_opcodestr[0x6f] = "if_scmple";
  jardin_jcvm_opcodestr[0x70] = "goto";
  jardin_jcvm_opcodestr[0x71] = "jsr";
  jardin_jcvm_opcodestr[0x72] = "ret";
  jardin_jcvm_opcodestr[0x73] = "stableswitch";
  jardin_jcvm_opcodestr[0x74] = "itableswitch";
  jardin_jcvm_opcodestr[0x75] = "slookupswitch";
  jardin_jcvm_opcodestr[0x76] = "ilookupswitch";
  jardin_jcvm_opcodestr[0x77] = "areturn";
  jardin_jcvm_opcodestr[0x78] = "sreturn";
  jardin_jcvm_opcodestr[0x79] = "ireturn";
  jardin_jcvm_opcodestr[0x7a] = "return";
  jardin_jcvm_opcodestr[0x7b] = "getstatic_a";
  jardin_jcvm_opcodestr[0x7c] = "getstatic_b";
  jardin_jcvm_opcodestr[0x7d] = "getstatic_s";
  jardin_jcvm_opcodestr[0x7e] = "getstatic_i";
  jardin_jcvm_opcodestr[0x7f] = "putstatic_a";
  jardin_jcvm_opcodestr[0x80] = "putstatic_b";
  jardin_jcvm_opcodestr[0x81] = "putstatic_s";
  jardin_jcvm_opcodestr[0x82] = "putstatic_i";
  jardin_jcvm_opcodestr[0x83] = "getfield_a";
  jardin_jcvm_opcodestr[0x84] = "getfield_b";
  jardin_jcvm_opcodestr[0x85] = "getfield_s";
  jardin_jcvm_opcodestr[0x86] = "getfield_i";
  jardin_jcvm_opcodestr[0x87] = "putfield_a";
  jardin_jcvm_opcodestr[0x88] = "putfield_b";
  jardin_jcvm_opcodestr[0x89] = "putfield_s";
  jardin_jcvm_opcodestr[0x8a] = "putfield_i";
  jardin_jcvm_opcodestr[0x8b] = "invokevirtual";
  jardin_jcvm_opcodestr[0x8c] = "invokespecial";
  jardin_jcvm_opcodestr[0x8d] = "invokestatic";
  jardin_jcvm_opcodestr[0x8e] = "invokeinterface";
  jardin_jcvm_opcodestr[0x8f] = "new";
  jardin_jcvm_opcodestr[0x90] = "newarray";
  jardin_jcvm_opcodestr[0x91] = "anewarray";
  jardin_jcvm_opcodestr[0x92] = "arraylength";
  jardin_jcvm_opcodestr[0x93] = "athrow";
  jardin_jcvm_opcodestr[0x94] = "checkcast";
  jardin_jcvm_opcodestr[0x95] = "instanceof";
  jardin_jcvm_opcodestr[0x96] = "sinc_w";
  jardin_jcvm_opcodestr[0x97] = "iinc_w";
  jardin_jcvm_opcodestr[0x98] = "ifeq_w";
  jardin_jcvm_opcodestr[0x99] = "ifne_w";
  jardin_jcvm_opcodestr[0x9a] = "iflt_w";
  jardin_jcvm_opcodestr[0x9b] = "ifge_w";
  jardin_jcvm_opcodestr[0x9c] = "ifgt_w";
  jardin_jcvm_opcodestr[0x9d] = "ifle_w";
  jardin_jcvm_opcodestr[0x9e] = "ifnull_w";
  jardin_jcvm_opcodestr[0x9f] = "ifnonull_w";
  jardin_jcvm_opcodestr[0xa0] = "if_acmpeq_w";
  jardin_jcvm_opcodestr[0xa1] = "if_acmpne_w";
  jardin_jcvm_opcodestr[0xa2] = "if_scmpeq_w";
  jardin_jcvm_opcodestr[0xa3] = "if_scmpne_w";
  jardin_jcvm_opcodestr[0xa4] = "if_scmplt_w";
  jardin_jcvm_opcodestr[0xa5] = "if_scmpge_w";
  jardin_jcvm_opcodestr[0xa6] = "if_scmpgt_w";
  jardin_jcvm_opcodestr[0xa7] = "if_scmple_w";
  jardin_jcvm_opcodestr[0xa8] = "goto_w";
  jardin_jcvm_opcodestr[0xa9] = "getfield_a_w";
  jardin_jcvm_opcodestr[0xaa] = "getfield_b_w";
  jardin_jcvm_opcodestr[0xab] = "getfield_s_w";
  jardin_jcvm_opcodestr[0xac] = "getfield_i_w";
  jardin_jcvm_opcodestr[0xad] = "getfield_a_this";
  jardin_jcvm_opcodestr[0xae] = "getfield_b_this";
  jardin_jcvm_opcodestr[0xaf] = "getfield_s_this";
  jardin_jcvm_opcodestr[0xb0] = "getfield_i_this";
  jardin_jcvm_opcodestr[0xb1] = "putfield_a_w";
  jardin_jcvm_opcodestr[0xb2] = "putfield_b_w";
  jardin_jcvm_opcodestr[0xb3] = "putfield_s_w";
  jardin_jcvm_opcodestr[0xb4] = "putfield_i_w";
  jardin_jcvm_opcodestr[0xb5] = "putfield_a_this";
  jardin_jcvm_opcodestr[0xb6] = "putfield_b_this";
  jardin_jcvm_opcodestr[0xb7] = "putfield_s_this";
  jardin_jcvm_opcodestr[0xb8] = "putfield_i_this";
  jardin_jcvm_opcodestr[0xfe] = "impdep1";
  jardin_jcvm_opcodestr[0xff] = "impdep2";
};

static void jardin_jcvm_jumpstart(struct jardin_jcvm_applet *context, struct jardin_jcvm_frame *invoker_frame, struct method_info *method_info);

#define JARDIN_STACK_SIZE     10000
static struct jardin_jcvm_frame jardin_jcvm_stack[JARDIN_STACK_SIZE];
static unsigned int jardin_jcvm_stack_index = 0;

struct jardin_jcvm_applet *jardin_jcvm_applets;
static unsigned int jardin_jcvm_applet_count;

void jardin_jcvm_applet_alloc(unsigned int applet_count) {
  if ((jardin_jcvm_applets = calloc(applet_count, sizeof(struct jardin_jcvm_applet))) == NULL) {
    JARDIN_FATAL("could not allocate %d x jardin_jcvm_applet", applet_count);
  }
  jardin_jcvm_applet_count = applet_count;
}


void jardin_jcvm_applet_free() {
  unsigned int i;

  for (i = 0; i < jardin_jcvm_applet_count; i++) {
    jardin_cap_free(&jardin_jcvm_applets[i].cap);
  }

  free(jardin_jcvm_applets);
}


void jardin_jcvm_init_static_field_image(struct jardin_jcvm_applet *applet) {
  int image_offset = 0;
  int i, j;
  
  if (applet->cap->static_field_comp.image_size == 0) {
    return;
  }

  if ((applet->static_field_image = calloc(sizeof(jardin_jcvm_word), applet->cap->static_field_comp.image_size)) == NULL) {
    JARDIN_FATAL("failed to calloc static_field_image");
  }
  
  // segment 1
  for(i = 0; i < applet->cap->static_field_comp.array_init_count; i++) {
    for (j = 0; j < applet->cap->static_field_comp.array_init[i].count; i++) {
      applet->static_field_image[image_offset++] = applet->cap->static_field_comp.array_init[i].values[j];
    }
  }
  // segement 2
  while ( image_offset < applet->cap->static_field_comp.reference_count) {
    applet->static_field_image[image_offset++] = 0;
  }
  // segment 3
  for(i = 0; i < applet->cap->static_field_comp.default_value_count; i++) {
    applet->static_field_image[image_offset++] = 0;
  }
  // segment 4
  for(i = 0; i < applet->cap->static_field_comp.non_default_value_count; i++) {
    applet->static_field_image[image_offset++] = applet->cap->static_field_comp.non_default_values[i];
  }
  JARDIN_DEBUG("image_size: %d  image_offset: %d", applet->cap->static_field_comp.image_size, image_offset);
  if (applet->cap->static_field_comp.image_size > 0 && applet->cap->static_field_comp.image_size != (image_offset + 1)) {
    JARDIN_FATAL("size mismatch while initializing static_field_image");
  }
}


void jardin_jcvm_init() {
  int i;
  
  for (i = 0; i < jardin_jcvm_applet_count; i++) {
    jardin_jcvm_init_static_field_image(&jardin_jcvm_applets[i]);
  }

  jardin_jcvm_init_opcodestr();
  
  memset(jardin_jcvm_stack, 0, sizeof(struct jardin_jcvm_frame) * JARDIN_STACK_SIZE);
}

void jardin_jcvm_free() {
  int i;
  
  for (i = 0; i < jardin_jcvm_applet_count; i++) {
    if (jardin_jcvm_applets[i].static_field_image != NULL) {
      free(jardin_jcvm_applets[i].static_field_image);
    }
  }
}


static struct jardin_jcvm_frame *jardin_jcvm_allocate_frame(int local_variables, int operand_stack, struct jardin_jcvm_applet *context) {
  struct jardin_jcvm_frame *frame;
  frame = &jardin_jcvm_stack[jardin_jcvm_stack_index++];
  if (local_variables > 0) {
    frame->local_variables = calloc(local_variables, sizeof(jardin_jcvm_word));
  }
  if (operand_stack > 0) {
    frame->operand_stack = calloc(operand_stack, sizeof(jardin_jcvm_word));
  }
  frame->operand_stack_index = 0;
  frame->context = context;
  frame->local_variables_size = local_variables;
  frame->operand_stack_size = operand_stack;
  return frame;
}

static void jardin_jcvm_release_frame(struct jardin_jcvm_frame *frame) {
  if (frame != &jardin_jcvm_stack[jardin_jcvm_stack_index - 1]) {
    JARDIN_FATAL("release frame is not top most frame");
  }
  if (frame->local_variables != NULL) {
    free(frame->local_variables);
  }
  if (frame->operand_stack != NULL) {
    free(frame->operand_stack);
  }
  jardin_jcvm_stack_index--;
  memset(&jardin_jcvm_stack[jardin_jcvm_stack_index], 0, sizeof(struct jardin_jcvm_frame));
}

static void jardin_jcvm_push_stack(struct jardin_jcvm_frame *frame, jardin_jcvm_word value) {
  JARDIN_DEBUG("push_stack[%d] = 0x%.4x", frame->operand_stack_index, value);
  if (frame->operand_stack_index >= frame->operand_stack_size) {
    JARDIN_FATAL("frame operand stack size (%d) exceeded", frame->operand_stack_size);
  }
  frame->operand_stack[frame->operand_stack_index++] = value;
}

static jardin_jcvm_word jardin_jcvm_pop_stack(struct jardin_jcvm_frame *frame) {
  JARDIN_DEBUG("pop_stack[%d] = 0x%.4x", frame->operand_stack_index - 1, frame->operand_stack[frame->operand_stack_index - 1] );
  if (frame->operand_stack_index <= 0) {
    JARDIN_FATAL("no more value on operand stack (%d)", frame->operand_stack_index);
  }
  return frame->operand_stack[--frame->operand_stack_index];
}

static jardin_jcvm_word jardin_jcvm_get_stack(struct jardin_jcvm_frame *frame, int offset) {
  JARDIN_DEBUG("get_stack[%d %d] = 0x%.4x", frame->operand_stack_index, offset, frame->operand_stack[frame->operand_stack_index + offset] );
  if ((frame->operand_stack_index + offset) < 0) {
    JARDIN_FATAL("get stack index not in range (%d)", frame->operand_stack_index + offset);
  }
  return frame->operand_stack[frame->operand_stack_index + offset];
}

static void jardin_jcvm_set_stack(struct jardin_jcvm_frame *frame, int offset, jardin_jcvm_word value) {
  JARDIN_DEBUG("set_stack[%d %d] = 0x%.4x", frame->operand_stack_index, offset, value );
  if ((frame->operand_stack_index + offset) <= 0) {
    JARDIN_FATAL("set stack index not in range (%d)", frame->operand_stack_index + offset);
  }
  frame->operand_stack[frame->operand_stack_index + offset] = value;
}

static void jardin_jcvm_set_local_variable(struct jardin_jcvm_frame *frame, int index, jardin_jcvm_word value) {
  if (index < 0 || index >= frame->local_variables_size) {
    JARDIN_FATAL("local varible index (%d) out of range", index);
  }
  frame->local_variables[index] = value;
}

static jardin_jcvm_word jardin_jcvm_get_local_variable(struct jardin_jcvm_frame *frame, int index) {
  if (index < 0 || index >= frame->local_variables_size) {
    JARDIN_FATAL("local varible index (%d) out of range", index);
  }
  return frame->local_variables[index];
}

#define T_BOOLEAN  10
#define T_BYTE     11
#define T_SHORT    12
#define T_INT      13

struct object_heap {
  enum { CLASS, ARRAY } type;
  union {
    struct {
      struct jardin_jcvm_applet  *applet;
      struct class_info *class_info;
    } class;
    struct {
      unsigned char atype;
      jardin_jcvm_short count;
    } array;
  } internal_data;
  int max_data;
  unsigned short data[10];
};

#define MAX_OBJECT_HEAP  10000

static struct object_heap object_heap[MAX_OBJECT_HEAP];
static u2 object_heap_index = 1;

jardin_jcvm_reference alloc_class(struct jardin_jcvm_applet *applet, struct class_info * class_info) {
  object_heap[object_heap_index].type = CLASS;
  object_heap[object_heap_index].internal_data.class.applet = applet;
  object_heap[object_heap_index].internal_data.class.class_info = class_info;
  return object_heap_index++;
};

jardin_jcvm_reference alloc_array(unsigned char atype, jardin_jcvm_short count) {
  object_heap[object_heap_index].type = ARRAY;
  object_heap[object_heap_index].internal_data.array.atype = atype;
  object_heap[object_heap_index].internal_data.array.count = count;
  return object_heap_index++;
};


static struct method_info *get_method_info_from_method_offset(struct jardin_cap *cap, u2 method_offset) {
  int i, j;
  for (i = 0; i < cap->descriptor_comp.class_count; i++) {
    for (j = 0; j < cap->descriptor_comp.classes[i].method_count; j++) {
      if (cap->descriptor_comp.classes[i].methods[j].method_offset == method_offset) {
	return cap->descriptor_comp.classes[i].methods[j].method_info;
      }
    }
  }
  JARDIN_FATAL("no method_info for this offset (0x%.04x) found.", method_offset);
}

static int get_class_index_from_offset(struct jardin_cap *cap, u2 class_offset) {
  int i;
  
  for (i = 0; i < cap->descriptor_comp.class_count; i++) {
    if (cap->class_comp.classes[i].cap_file_offset == class_offset +3) {
      return i;
    }
  }
  JARDIN_FATAL("no matching class for class_offset 0x%.04x found", class_offset);
}

static int compare_AID(u1 *aAID, u1 *bAID, u1 AID_length) {
  int i;
  for (i = 0; i < AID_length; i++) {
    if (aAID[i] != bAID[i]) {
      return 0;
    }
  }
  return 1;
}

static int get_applet_index(u1 *AID, u1 AID_length) {
  int i;
  
  for (i = 0; i < jardin_jcvm_applet_count; i++) {
    if (jardin_jcvm_applets[i].cap->header_comp.this_package.AID_length == AID_length &&
	compare_AID(AID, jardin_jcvm_applets[i].cap->header_comp.this_package.AID, AID_length)) {
      return i;
    }
  }
  JARDIN_FATAL("no matching applet found");
}

static int get_class_index(struct descriptor_component *descriptor_comp, u1 class_token) {
  int i;
  
  for (i = 0; i < descriptor_comp->class_count; i++) {
    if (descriptor_comp->classes[i].token == class_token) {
      return i;
    }
  }
  JARDIN_FATAL("no matching class found");
}

static void invoke_virtual_method(struct jardin_jcvm_applet *applet, struct jardin_jcvm_frame *frame, u1 info0, u1 info1, u1 token, int follow_object) {
  // internal_class_ref case
  u2 internal_class_ref =  ( info0 << 8 ) | info1;

  // external_class_ref case
  u1 external_package_token = info0 & 0x7f;
  u1 external_class_token = info1;

  u2 method_offset;
  
  int i;
	
  printf("called invoke_virtual_method(inf0=0x%.02x, info1=0x%.02x, token=0x%.02x,)\n", info0, info1, token);
  
  if (token & 0x80) {
    printf(" token high bit is one, indicating a package-visible method reference.\n");
  } else {
    printf(" token high bit is zero, indicating public/protected method reference.\n");
  }

  if (info0 == 0xff || info1 == 0xff) {
    fprintf(stderr, "Error: super_class_ref == 0xffff.\n");
    exit(1);
  }

  if (internal_class_ref <= 32767) {
    int class_index = get_class_index_from_offset(applet->cap, internal_class_ref);
    
    if (token < applet->cap->class_comp.classes[class_index].public_method_table_base) {
      printf(" virtual_methodref.token < public_method_table_base.\n");
      printf(" following superclass ref:\n");
      invoke_virtual_method(applet, frame, (applet->cap->class_comp.classes[class_index].super_class_ref.internal_class_ref & 0xff00) >> 8, applet->cap->class_comp.classes[class_index].super_class_ref.internal_class_ref & 0x00ff, token, follow_object);
    }
    method_offset = applet->cap->class_comp.classes[class_index]. public_virtual_method_table[token - applet->cap->class_comp.classes[class_index].public_method_table_base];
    for (i = 0; i < applet->cap->method_comp.method_count; i++) {
      if (applet->cap->method_comp.methods[i].cap_file_offset == method_offset) {
	// XXXXX check for right method type
	jardin_jcvm_jumpstart(applet, frame, &applet->cap->method_comp.methods[i]);
	return;
      }
    }
    if (i == applet->cap->method_comp.method_count) {
      fprintf(stderr, "Error: no method_info found with offset 0x%.04x.\n", method_offset);
      exit(1);
    }
  } else {
    // external_class_ref
    int applet_index;
    int desc_comp_class_index;
    int class_index;
    u2  class_offset;
    
    printf(" external class_ref (package_token=%d, class_token=%d) :\n", external_package_token, external_class_token);
    if (external_package_token >= applet->cap->import_comp.count) {
      fprintf(stderr, "Error: external_package_token >= applet->import_comp.count\n");
      exit(1);
    }
    applet_index = get_applet_index(applet->cap->import_comp.packages[external_package_token].AID, applet->cap->import_comp.packages[external_package_token].AID_length);
    printf(" applet_index=%d\n", applet_index);
    desc_comp_class_index = get_class_index(&jardin_jcvm_applets[applet_index].cap->descriptor_comp, external_class_token);
    printf(" desc_comp_class_index=%d\n", desc_comp_class_index);
    class_offset = jardin_jcvm_applets[applet_index].cap->descriptor_comp.classes[desc_comp_class_index].this_class_ref.internal_class_ref;
    printf(" class_offset=0x%.4x\n", class_offset);
    
    class_index = get_class_index_from_offset(jardin_jcvm_applets[applet_index].cap, class_offset);
    printf(" class_comp.class_info.public_method_table_base=%d\n", jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].public_method_table_base);
    printf(" class_comp.class_info.public_method_table_count=%d\n", jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].public_method_table_count);
    //
    if (token < jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].public_method_table_base) {
      printf(" virtual_methodref.token < public_method_table_base.\n");
      printf(" following superclass ref:\n");
      invoke_virtual_method(&jardin_jcvm_applets[applet_index], frame, (jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].super_class_ref.internal_class_ref & 0xff00) >> 8, jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].super_class_ref.internal_class_ref & 0x00ff, token, follow_object);
    }
    method_offset = jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].public_virtual_method_table[token - jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index].public_method_table_base];
    printf(" method_offset: 0x%.4x\n", method_offset);
    for (i = 0; i < jardin_jcvm_applets[applet_index].cap->method_comp.method_count; i++) {
      if (jardin_jcvm_applets[applet_index].cap->method_comp.methods[i].cap_file_offset == method_offset) {
	if (follow_object) {
	  int nargs = METHOD_HEADER_INFO_NARGS(jardin_jcvm_applets[applet_index].cap->method_comp.methods[i].method_header);
	  jardin_jcvm_reference objectref= jardin_jcvm_get_stack(frame, -nargs);
	  printf(" objectref: %d\n", objectref);
	  if (token < object_heap[objectref].internal_data.class.class_info->public_method_table_base) {
	    printf(" virtual_methodref.token < objectref->public_method_table_base.\n");
	    printf(" following superclass ref:\n");
	    invoke_virtual_method(object_heap[objectref].internal_data.class.applet, frame, (object_heap[objectref].internal_data.class.class_info->super_class_ref.internal_class_ref & 0xff00) >> 8, object_heap[objectref].internal_data.class.class_info->super_class_ref.internal_class_ref & 0x00ff, token, 0);
	    return;
	  }
	  method_offset = object_heap[objectref].internal_data.class.class_info->public_virtual_method_table[token - object_heap[objectref].internal_data.class.class_info->public_method_table_base];
	  for (i = 0; i <  object_heap[objectref].internal_data.class.applet->cap->method_comp.method_count; i++) {
	    if (object_heap[objectref].internal_data.class.applet->cap->method_comp.methods[i].cap_file_offset == method_offset) {
	      jardin_jcvm_jumpstart(object_heap[objectref].internal_data.class.applet, frame, &object_heap[objectref].internal_data.class.applet->cap->method_comp.methods[i]);
	      return;
	    }
	  }
	  if (i == object_heap[objectref].internal_data.class.applet->cap->method_comp.method_count) {
	    fprintf(stderr, "Error: no method_info found with offset 0x%.04x.\n", method_offset);
	    exit(1);
	  }
	} else {
	  jardin_jcvm_jumpstart(&jardin_jcvm_applets[applet_index], frame, &jardin_jcvm_applets[applet_index].cap->method_comp.methods[i]);
	  return;
	}
      }
      //
    }
  }
  fprintf(stderr, "Error: should not be here.....\n");
  exit(1);
}

static int get_method_index(struct class_descriptor_info *class_descriptor_info, u1 method_token, u1 access_flags) {
  int i;
  
  for (i = 0; i < class_descriptor_info->method_count; i++) {
    if (class_descriptor_info->methods[i].token == method_token &&
	(class_descriptor_info->methods[i].access_flags & access_flags)) {
      return i;
    }
  }
  JARDIN_FATAL("no matching method found");
}

static void invoke_external_method(struct jardin_jcvm_applet *applet, struct jardin_jcvm_frame *invoker_frame, struct static_methodref_info static_method_ref, u1 access_flags) {
  int package_index;
  int applet_index;
  int class_index;
  int method_index;
  struct method_info *method_info;
  int i;
  
  printf("invoke_external_method (package_token=0x%.02x, class_token=0x%.02x, token=0x%.02x).\n", static_method_ref.static_field_ref.external_ref.package_token, static_method_ref.static_field_ref.external_ref.class_token, static_method_ref.static_field_ref.external_ref.token);

  package_index = static_method_ref.static_field_ref.external_ref.package_token & 0x7f;
  printf(" AID:");
  for (i = 0; i < applet->cap->import_comp.packages[package_index].AID_length; i++) {
    printf(" 0x%.02x", applet->cap->import_comp.packages[package_index].AID[i]);
  }
  printf("\n");
  
  applet_index = get_applet_index(applet->cap->import_comp.packages[package_index].AID, applet->cap->import_comp.packages[package_index].AID_length);
  printf(" applet_index = %d\n", applet_index);
  class_index = get_class_index(&jardin_jcvm_applets[applet_index].cap->descriptor_comp, static_method_ref.static_field_ref.external_ref.class_token);
  printf(" class_index = %d\n", class_index);
  method_index = get_method_index(&jardin_jcvm_applets[applet_index].cap->descriptor_comp.classes[class_index], static_method_ref.static_field_ref.external_ref.token, access_flags);
  printf(" method_index = %d\n", method_index);
  printf(" method_offset = 0x%.4d\n", jardin_jcvm_applets[applet_index].cap->descriptor_comp.classes[class_index].methods[method_index].method_offset);
  method_info = get_method_info_from_method_offset(jardin_jcvm_applets[applet_index].cap, jardin_jcvm_applets[applet_index].cap->descriptor_comp.classes[class_index].methods[method_index].method_offset);

  jardin_jcvm_jumpstart(&jardin_jcvm_applets[applet_index], invoker_frame, method_info);
  
  return;
}

static u2 create_new_object(struct jardin_jcvm_frame *frame, u2 index) {
  struct classref_info classref_info;
  int class_index;

  if (frame->context->cap->constant_pool_comp.constant_pool[index].tag != CONSTANT_Classref) {
    fprintf(stderr, "Error: cp_info.tag != CONSTANT_Classref.\n");
    exit(1);
  }
  
  classref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[index].tag;
  classref_info.class_ref.internal_class_ref =  (frame->context->cap->constant_pool_comp.constant_pool[index].info[0] << 8) | frame->context->cap->constant_pool_comp.constant_pool[index].info[1];
  classref_info.padding = frame->context->cap->constant_pool_comp.constant_pool[index].info[2];

  if (classref_info.class_ref.internal_class_ref > 32767) {
    int package_index;
    int applet_index;
    int class_desc_index;
    
    u2 class_offset;
    int i;
    u1 package_token = frame->context->cap->constant_pool_comp.constant_pool[index].info[0];
    u2 class_token = frame->context->cap->constant_pool_comp.constant_pool[index].info[1];
    printf(" external_class_ref (package_token=0x%.02x, class_token=0x%.02x):\n", package_token, class_token);
    package_index = package_token & 0x7f;
    printf(" AID:");
    for (i = 0; i < frame->context->cap->import_comp.packages[package_index].AID_length; i++) {
      printf(" 0x%.02x", frame->context->cap->import_comp.packages[package_index].AID[i]);
    }
    printf("\n");
    applet_index = get_applet_index(frame->context->cap->import_comp.packages[package_index].AID, frame->context->cap->import_comp.packages[package_index].AID_length);
    class_desc_index = get_class_index(&jardin_jcvm_applets[applet_index].cap->descriptor_comp, class_token);
    
    class_offset = jardin_jcvm_applets[applet_index].cap->descriptor_comp.classes[class_desc_index].this_class_ref.internal_class_ref;
    
    class_index = get_class_index_from_offset(jardin_jcvm_applets[applet_index].cap, class_offset);
    if (CLASS_INFO_FLAGS(jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index]) & ACC_INTERFACE) {
    fprintf(stderr, "Error: class_info is interface.\n");
    exit(1);
  }
  return alloc_class(&jardin_jcvm_applets[applet_index], &jardin_jcvm_applets[applet_index].cap->class_comp.classes[class_index]);
  }

  class_index = get_class_index_from_offset(frame->context->cap, classref_info.class_ref.internal_class_ref);
  if (CLASS_INFO_FLAGS(frame->context->cap->class_comp.classes[class_index]) & ACC_INTERFACE) {
    fprintf(stderr, "Error: class_info is interface.\n");
    exit(1);
  }
  return alloc_class(frame->context, &frame->context->cap->class_comp.classes[class_index]);
}


#define DEBUG_NULL_BYTE printf("bytecode[0x%.02x]: 0x%.02x    %s;\n", frame->pc, method_info->bytecodes[frame->pc], jardin_jcvm_opcodestr[method_info->bytecodes[frame->pc]])
#define DEBUG_ONE_BYTE printf("bytecode[0x%.02x]: 0x%.02x    %s %d;\n", frame->pc, method_info->bytecodes[frame->pc], jardin_jcvm_opcodestr[method_info->bytecodes[frame->pc]], method_info->bytecodes[frame->pc+1])
#define DEBUG_TWO_BYTE printf("bytecode[0x%.02x]: 0x%.02x    %s %d;\n", frame->pc, method_info->bytecodes[frame->pc], jardin_jcvm_opcodestr[method_info->bytecodes[frame->pc]], (method_info->bytecodes[frame->pc+1] << 8) | method_info->bytecodes[frame->pc+2])
#define DEBUG_TWO_ONE_BYTE printf("bytecode[0x%.02x]: 0x%.02x    %s 0x%.2x 0x%.2x;\n", frame->pc, method_info->bytecodes[frame->pc], jardin_jcvm_opcodestr[method_info->bytecodes[frame->pc]], method_info->bytecodes[frame->pc+1], method_info->bytecodes[frame->pc+2])

#define ONE_BYTE_VALUE (method_info->bytecodes[frame->pc+1])
#define TWO_BYTE_VALUE ((method_info->bytecodes[frame->pc+1] << 8) | method_info->bytecodes[frame->pc+2])

static void nyi(struct method_info *method_info, struct jardin_jcvm_frame *frame) {
  DEBUG_NULL_BYTE;
  JARDIN_FATAL("Error: Opcode not yet implemented.\n");
}
static void jardin_jcvm_jumpstart(struct jardin_jcvm_applet *context, struct jardin_jcvm_frame *invoker_frame, struct method_info *method_info) {
  int i;
  struct jardin_jcvm_frame *frame = jardin_jcvm_allocate_frame(METHOD_HEADER_INFO_MAX_LOCALS(method_info->method_header) + METHOD_HEADER_INFO_NARGS(method_info->method_header), METHOD_HEADER_INFO_MAX_STACK(method_info->method_header), context);
  frame->pc = 0;
  frame->invoker_frame = invoker_frame;
  
  JARDIN_DEBUG("     .stack %d;\n", frame->operand_stack_size);
  JARDIN_DEBUG("     .locals %d;\n", frame->local_variables_size);
  JARDIN_DEBUG("method_header->nargs: %d\n", METHOD_HEADER_INFO_NARGS(method_info->method_header));
  for (i = 0; i < METHOD_HEADER_INFO_NARGS(method_info->method_header); i++) {
    jardin_jcvm_set_local_variable(frame, METHOD_HEADER_INFO_NARGS(method_info->method_header)-i -1, jardin_jcvm_pop_stack(invoker_frame));
  }

  /*
  if (frame.local_vars[0] == 0) {
    fprintf(stderr, "Error: objectref (frame.local_vars[0]) == 0.\n");
    // XXXXX throw NullPointerException
  }
  */

  while(1) {
    switch(method_info->bytecodes[frame->pc]) {
    case OPCODE_nop:
      DEBUG_NULL_BYTE;
      frame->pc += 1;
      break;
    case OPCODE_aconst_null:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 0);
      frame->pc += 1;
      break;
    case OPCODE_sconst_m1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, -1);
      frame->pc += 1;
      break;
    case OPCODE_sconst_0:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 0);
      frame->pc += 1;
      break;
    case OPCODE_sconst_1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 1);
      frame->pc += 1;
      break;
    case OPCODE_sconst_2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 2);
      frame->pc += 1;
      break;
    case OPCODE_sconst_3:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 3);
      frame->pc += 1;
      break;
    case OPCODE_sconst_4:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 4);
      frame->pc += 1;
      break;
    case OPCODE_sconst_5:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, 5);
      frame->pc += 1;
      break;
    case OPCODE_iconst_m1:
    case OPCODE_iconst_0:
    case OPCODE_iconst_1:
    case OPCODE_iconst_2:
    case OPCODE_iconst_3:
    case OPCODE_iconst_4:
    case OPCODE_iconst_5:
       nyi(method_info, frame);
      break;
    case OPCODE_bspush:
      DEBUG_ONE_BYTE;
      {
	jardin_jcvm_short short_value = ONE_BYTE_VALUE;
	jardin_jcvm_push_stack(frame, short_value);
	frame->pc += 2;
      }
      break;
    case OPCODE_sspush:
      DEBUG_TWO_BYTE;
      jardin_jcvm_push_stack(frame, TWO_BYTE_VALUE);
      frame->pc += 3;
      break;
    case OPCODE_bipush:
    case OPCODE_sipush:
    case OPCODE_iipush:
      nyi(method_info, frame);
      break;
    case OPCODE_aload:
      DEBUG_ONE_BYTE;
      {
	int index = ONE_BYTE_VALUE;
	jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, index));
	frame->pc += 2;
      }
      break;
    case OPCODE_sload:
      DEBUG_ONE_BYTE;
      {
	int index = ONE_BYTE_VALUE;
	jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, index));
	frame->pc += 2;
      }
      break;
    case OPCODE_iload:
      nyi(method_info, frame);
      break;
    case OPCODE_aload_0:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 0));
      frame->pc += 1;
      break;
    case OPCODE_aload_1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 1));
      frame->pc += 1;
      break;
    case OPCODE_aload_2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 2));
      frame->pc += 1;
      break;
    case OPCODE_aload_3:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 3));
      frame->pc += 1;
      break;
    case OPCODE_sload_0:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 0));
      frame->pc += 1;
      break;
    case OPCODE_sload_1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 1));
      frame->pc += 1;
      break;
    case OPCODE_sload_2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 2));
      frame->pc += 1;
      break;
    case OPCODE_sload_3:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_local_variable(frame, 3));
      frame->pc += 1;
      break;
    case OPCODE_iload_0:
    case OPCODE_iload_1:
    case OPCODE_iload_2:
    case OPCODE_iload_3:
    case OPCODE_aaload:
      nyi(method_info, frame);
      break; 
    case OPCODE_baload:
      DEBUG_NULL_BYTE;
      {
	jardin_jcvm_short index = jardin_jcvm_pop_stack(frame);
	u2 arrayref = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_short value;
	
	if (arrayref == 0) {
	  fprintf(stderr, "Error: arrayref == 0, throw NullPointerException not yet implemented.\n");
	  exit(1);
	}
	if (object_heap[arrayref].type != ARRAY) {
	  fprintf(stderr, "Error: arrayref referes not to array object.\n");
	  exit(0);
	}
	if (object_heap[arrayref].internal_data.array.atype != T_BYTE) {
	  fprintf(stderr, "Error: arrayref referes not to byte array object.\n");
	  exit(0);
	}
	if (index >= object_heap[arrayref].internal_data.array.count) {
	  fprintf(stderr, "Error: throw ArrayIndexOutOfBounds exception not yet implemented.");
	}
	value = object_heap[arrayref].data[index];
	jardin_jcvm_push_stack(frame, value);;
	frame->pc += 1;
      }
      break;
    case OPCODE_saload:
    case OPCODE_iaload:
    case OPCODE_astore:
      nyi(method_info, frame);
      break; 
    case OPCODE_sstore:
      DEBUG_ONE_BYTE;
      {
	int index = ONE_BYTE_VALUE;
	jardin_jcvm_set_local_variable(frame, index, jardin_jcvm_pop_stack(frame));
	frame->pc += 2;
      }
      break;
    case OPCODE_istore:
      nyi(method_info, frame);
      break; 
    case OPCODE_astore_0:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 0, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_astore_1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 1, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_astore_2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 2, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_astore_3:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 3, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_sstore_0:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 0, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_sstore_1:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 1, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_sstore_2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 2, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_sstore_3:
      DEBUG_NULL_BYTE;
      jardin_jcvm_set_local_variable(frame, 3, jardin_jcvm_pop_stack(frame));
      frame->pc += 1;
      break;
    case OPCODE_istore_0:
    case OPCODE_istore_1:
    case OPCODE_istore_2:
    case OPCODE_istore_3:
    case OPCODE_aastore:
      nyi(method_info, frame);
      break;
    case OPCODE_bastore:
      DEBUG_NULL_BYTE;
      {
	jardin_jcvm_short value = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_short index = jardin_jcvm_pop_stack(frame);
	u2 arrayref = jardin_jcvm_pop_stack(frame);
	
	if (arrayref == 0) {
	  fprintf(stderr, "Error: arrayref == 0, throw NullPointerException not yet implemented.\n");
	  exit(1);
	}
	if (object_heap[arrayref].type != ARRAY) {
	  fprintf(stderr, "Error: arrayref referes not to array object.\n");
	  exit(0);
	}
	if (object_heap[arrayref].internal_data.array.atype != T_BYTE) {
	  fprintf(stderr, "Error: arrayref referes not to byte array object.\n");
	  exit(0);
	}
	if (index >= object_heap[arrayref].internal_data.array.count) {
	  fprintf(stderr, "Error: throw ArrayIndexOutOfBounds exception not yet implemented.");
	}
	object_heap[arrayref].data[index] = value & 0x00ff;
	frame->pc += 1;
      }
      break;
    case OPCODE_sastore:
    case OPCODE_iastore:
      nyi(method_info, frame);
      break;
    case OPCODE_pop:
      DEBUG_NULL_BYTE;
      jardin_jcvm_pop_stack(frame);
      frame->pc += 1;
      break;
    case OPCODE_pop2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_pop_stack(frame);
      jardin_jcvm_pop_stack(frame);
      frame->pc += 1;
      break;
    case OPCODE_dup:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_stack(frame, -1));
      frame->pc += 1;
      break;
    case OPCODE_dup2:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_stack(frame, -2));
      jardin_jcvm_push_stack(frame, jardin_jcvm_get_stack(frame, -2));
      frame->pc += 1;
      break;
    case OPCODE_dup_x:
      DEBUG_ONE_BYTE;
      {
	unsigned char mn;
	unsigned char m;
	unsigned char n;
	unsigned char i;
        mn = ONE_BYTE_VALUE;
	m = (mn & 0xf0) >> 4;
	n = (mn & 0x0f);
	// XXXXX check: 1 <= m <= 4
	// XXXXX check: n == 0 || m <= n <= m+4
	if (n != 0) {
	  for (i = 0; i < m; i++) {
	    jardin_jcvm_push_stack(frame, 0);
	  }
	  for (i = 0; i < n; i++) {
	    jardin_jcvm_set_stack(frame, -i, jardin_jcvm_get_stack(frame, -i-m));
	  }
	  for (i = 0; i < m; i++) {
	    jardin_jcvm_set_stack(frame, -i -m -n, jardin_jcvm_get_stack(frame, -i-m));
	  }
	} else {
	  for (i = 0; i < m; i++) {
	    jardin_jcvm_push_stack(frame, jardin_jcvm_get_stack(frame, -m));
	  }
	} 
	frame->pc += 2;
      }
      break;
    case OPCODE_swap_x:
      nyi(method_info, frame);
      break;
    case OPCODE_sadd:
      DEBUG_NULL_BYTE;
      {
	signed short value2 = jardin_jcvm_pop_stack(frame);
	signed short value1 = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_push_stack(frame, value1 + value2);
	frame->pc += 1;
      }
      break;
    case OPCODE_iadd:
    case OPCODE_ssub:
    case OPCODE_isub:
    case OPCODE_smul:
    case OPCODE_imul:
    case OPCODE_sdiv:
    case OPCODE_idiv:
    case OPCODE_srem:
    case OPCODE_irem:
    case OPCODE_sneg:
    case OPCODE_ineg:
    case OPCODE_sshl:
    case OPCODE_ishl:
    case OPCODE_sshr:
    case OPCODE_ishr:
    case OPCODE_sushr:
    case OPCODE_iushr:
    case OPCODE_sand:
    case OPCODE_iand:
    case OPCODE_sor:
    case OPCODE_ior:
    case OPCODE_sxor:
    case OPCODE_ixor:
      nyi(method_info, frame);
      break;
    case OPCODE_sinc:
      DEBUG_TWO_ONE_BYTE;
      {
	unsigned char index = ONE_BYTE_VALUE;
	signed char const_val = ONE_BYTE_VALUE;
	jardin_jcvm_short tmp;
	printf("index: %d const: %d\n", index, const_val);
	tmp = jardin_jcvm_get_local_variable(frame, index);
	tmp += const_val;
	jardin_jcvm_set_local_variable(frame, index, tmp);
	frame->pc += 3;
      }
      break;
    case OPCODE_iinc:
    case OPCODE_s2b:
    case OPCODE_s2i:
    case OPCODE_i2b:
    case OPCODE_i2s:
    case OPCODE_icmp:
    case OPCODE_ifeq:
    case OPCODE_ifne:
      nyi(method_info, frame);
      break;
    case OPCODE_iflt:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	signed short value  = jardin_jcvm_pop_stack(frame);
	printf("branch %d value %d\n", branch, value);
	if (value < 0) {
	  frame->pc += branch; 
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_ifge:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	signed short value  = jardin_jcvm_pop_stack(frame);
	printf("branch %d value %d\n", branch, value);
	if (value >= 0) {
	  frame->pc += branch; 
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_ifgt:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	signed short value  = jardin_jcvm_pop_stack(frame);
	printf("branch %d value %d\n", branch, value);
	if (value > 0) {
	  frame->pc += branch; 
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_ifle:
      nyi(method_info, frame);
      break;
    case OPCODE_ifnull:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
        jardin_jcvm_reference value  = jardin_jcvm_pop_stack(frame);
	printf("branch %d objectref %d\n", branch, value);
	if (value == 0) {
	  frame->pc += branch; 
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_ifnonull:
       DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	jardin_jcvm_reference value  = jardin_jcvm_pop_stack(frame);
	printf("branch %d objectref %d\n", branch, value);
	if (value != 0) {
	  frame->pc += branch; 
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_if_acmpeq:
    case OPCODE_if_acmpne:
    case OPCODE_if_scmpeq:
    case OPCODE_if_scmpne:
      nyi(method_info, frame);
      break;
    case OPCODE_if_scmplt:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	jardin_jcvm_short value2 = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_short value1 = jardin_jcvm_pop_stack(frame);
	printf(" value1: %d value2: %d\n", value1, value2);
	if (value1 < value2) {
	  frame->pc += branch;
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_if_scmpge:
    case OPCODE_if_scmpgt:
      nyi(method_info, frame);
      break;
    case OPCODE_if_scmple:
      DEBUG_ONE_BYTE;
      {
	signed char branch = ONE_BYTE_VALUE;
	jardin_jcvm_short value2 = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_short value1 = jardin_jcvm_pop_stack(frame);
	printf(" value1: %d value2: %d\n", value1, value2);
	if (value1 <= value2) {
	  frame->pc += branch;
	} else {
	  frame->pc += 2;
	}
      }
      break;
    case OPCODE_goto:
      DEBUG_ONE_BYTE;
      {
	int branch = ONE_BYTE_VALUE;
	frame->pc += branch;
      }
      break;
    case OPCODE_jsr:
    case OPCODE_ret:
    case OPCODE_stableswitch:
    case OPCODE_itableswitch:
    case OPCODE_slookupswitch:
    case OPCODE_ilookupswitch:
      nyi(method_info, frame);
      break;
    case OPCODE_areturn:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame->invoker_frame, jardin_jcvm_pop_stack(frame));
      jardin_jcvm_release_frame(frame);
      return;
      break;
    case OPCODE_sreturn:
      DEBUG_NULL_BYTE;
      jardin_jcvm_push_stack(frame->invoker_frame, jardin_jcvm_pop_stack(frame));
      jardin_jcvm_release_frame(frame);
      return;
      break;
    case OPCODE_ireturn:
      nyi(method_info, frame);
    case OPCODE_return:
      DEBUG_NULL_BYTE;
      jardin_jcvm_release_frame(frame);
      return;
    case OPCODE_getstatic_a:
#define GET_STATIC_FIELD_IMAGE_A(a,b) ((a[b]<<8)|(a[b+1]))
#define PUT_STATIC_FIELD_IMAGE_A(a,b,c) (a[b]=(c&0xff00)>>8,(a[b+1]=c&0x00ff))
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;
	struct static_fieldref_info static_fieldref_info;
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_StaticFieldref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_StaticFieldref.\n", constant_pool_index);
	  exit(1);
	}
	static_fieldref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	static_fieldref_info.static_field_ref.internal_ref.padding =  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	static_fieldref_info.static_field_ref.internal_ref.offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	jardin_jcvm_push_stack(frame, GET_STATIC_FIELD_IMAGE_A(frame->context->static_field_image, static_fieldref_info.static_field_ref.internal_ref.offset));

	frame->pc += 3;
      }
      break;
    case OPCODE_getstatic_b:
      nyi(method_info, frame);
      break;
    case OPCODE_getstatic_s:
#define GET_STATIC_FIELD_IMAGE_S(a,b) (a[b])
#define PUT_STATIC_FIELD_IMAGE_S(a,b,c) (a[b]=c)
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;
	struct static_fieldref_info static_fieldref_info;
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_StaticFieldref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_StaticFieldref.\n", constant_pool_index);
	  exit(1);
	}
	static_fieldref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	static_fieldref_info.static_field_ref.internal_ref.padding =  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	static_fieldref_info.static_field_ref.internal_ref.offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	jardin_jcvm_push_stack(frame, GET_STATIC_FIELD_IMAGE_S(frame->context->static_field_image, static_fieldref_info.static_field_ref.internal_ref.offset));
	
	frame->pc += 3;
      }
      break;
    case OPCODE_getstatic_i:
      nyi(method_info, frame);
      break;
    case OPCODE_putstatic_a:
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;
	jardin_jcvm_word value = jardin_jcvm_pop_stack(frame);
	struct static_fieldref_info static_fieldref_info;
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_StaticFieldref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_StaticFieldref.\n", constant_pool_index);
	  exit(1);
	}
	static_fieldref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	static_fieldref_info.static_field_ref.internal_ref.padding =  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	static_fieldref_info.static_field_ref.internal_ref.offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	PUT_STATIC_FIELD_IMAGE_A(frame->context->static_field_image, static_fieldref_info.static_field_ref.internal_ref.offset, value);
	frame->pc += 3;
      }
      break;
    case OPCODE_putstatic_b:
    case OPCODE_putstatic_s:
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;
	jardin_jcvm_word value = jardin_jcvm_pop_stack(frame);
	struct static_fieldref_info static_fieldref_info;
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_StaticFieldref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_StaticFieldref.\n", constant_pool_index);
	  exit(1);
	}
	static_fieldref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	static_fieldref_info.static_field_ref.internal_ref.padding =  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	static_fieldref_info.static_field_ref.internal_ref.offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	PUT_STATIC_FIELD_IMAGE_S(frame->context->static_field_image, static_fieldref_info.static_field_ref.internal_ref.offset, value);
	frame->pc += 3;
      }
      break;
    case OPCODE_putstatic_i:
    case OPCODE_getfield_a:
    case OPCODE_getfield_b:
    case OPCODE_getfield_s:
    case OPCODE_getfield_i:
      nyi(method_info, frame);
      break;
    case OPCODE_putfield_a:
      DEBUG_ONE_BYTE;
      {
	struct instance_fieldref instance_fieldref;
	unsigned char index = ONE_BYTE_VALUE;
	jardin_jcvm_word value = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_reference objectref = jardin_jcvm_pop_stack(frame);
	if (frame->context->cap->constant_pool_comp. constant_pool[index].tag != CONSTANT_InstanceFieldref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_InstanceFieldref\n", index);
	  exit(1);
	}
	instance_fieldref.tag = frame->context->cap->constant_pool_comp.constant_pool[index].tag;
	instance_fieldref.class.internal_class_ref = (frame->context->cap->constant_pool_comp. constant_pool[index].info[0] << 8 ) | frame->context->cap->constant_pool_comp.constant_pool[index].info[1];
	instance_fieldref.token = frame->context->cap->constant_pool_comp.constant_pool[index].info[2];
	if (instance_fieldref.class.internal_class_ref > 32767) {
	  fprintf(stderr, "Eroor: instance_fieldref.class.external_class_ref not yet implemented.\n");
	} else {
	  object_heap[objectref].data[instance_fieldref.token - frame->context->cap->class_comp.classes[instance_fieldref.class.internal_class_ref].first_reference_token] = value;
	}
	frame->pc += 2;
      }
      break;
    case OPCODE_putfield_b:
    case OPCODE_putfield_s:
    case OPCODE_putfield_i:
      nyi(method_info, frame);
      break;
    case OPCODE_invokevirtual:
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;

	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_VirtualMethodref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_VirtualMethodref.\n", constant_pool_index);
	  exit(1);
	}
                     
	invoke_virtual_method(frame->context, frame, frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0], frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1],  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2], 1);
	frame->pc += 3;
      }
      break;
    case OPCODE_invokespecial:
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index;
	constant_pool_index = TWO_BYTE_VALUE;
	switch (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag) {
	case CONSTANT_StaticMethodref:
	  {
	    int i;
	    struct static_methodref_info static_methodref_info;
	    switch (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0]) {
	    case 0x00:
	      static_methodref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	      static_methodref_info.static_field_ref.internal_ref.padding = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	      static_methodref_info.static_field_ref.internal_ref.offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	      for (i = 0; i < frame->context->cap->method_comp.method_count; i++) {
		if (frame->context->cap->method_comp.methods[i].cap_file_offset == static_methodref_info.static_field_ref.internal_ref.offset) {
		  // XXXXX check for right method type
		  jardin_jcvm_jumpstart(frame->context, frame, &frame->context->cap->method_comp.methods[i]);
		  break;
		}
	      }
	      if (i == frame->context->cap->method_comp.method_count) {
		fprintf(stderr, "Error: no method_info found with offset 0x%.04x.\n", static_methodref_info.static_field_ref.internal_ref.offset);
		exit(1);
	      }
	      break;
	    default:
	      static_methodref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	      static_methodref_info.static_field_ref.external_ref.package_token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	      static_methodref_info.static_field_ref.external_ref.class_token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1];
	      static_methodref_info.static_field_ref.external_ref.token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	      invoke_external_method(frame->context, frame, static_methodref_info, ACC_INIT);
	    }
	  }
	  break;
	case CONSTANT_SuperMethodref:
	  fprintf(stderr, "Error: SuperMethodref.\n");
	  exit(1);
	  break;
	default:
	  fprintf(stderr, "Error: invokespecial for cp_info.tag 0x%.02x failed\n", frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag);
	  exit(1);
	}
	frame->pc += 3;
      }
      break;
    case OPCODE_invokestatic:
      DEBUG_TWO_BYTE;
      {
	u2 constant_pool_index = TWO_BYTE_VALUE;
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_StaticMethodref) {
	  fprintf(stderr, "Error: constant_pool_comp.constant_pool[0x%.02x].tag != CONSTANT_StaticMethodref.\n", constant_pool_index);
	  exit(1);
	}
	printf("info0: 0x%.2x info1: 0x%.2x info2: 0x%.2x\n", frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0],  frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1], frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2]);
	
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0] & 0x80) {
	  // external static method
	  struct static_methodref_info static_methodref_info;
	  static_methodref_info.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	  static_methodref_info.static_field_ref.external_ref.package_token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[0];
	  static_methodref_info.static_field_ref.external_ref.class_token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1];
	  static_methodref_info.static_field_ref.external_ref.token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	  invoke_external_method(frame->context, frame, static_methodref_info, ACC_STATIC);
	  
	} else {
	  // internal static method
	  u2 method_offset = (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1] << 8 ) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	  for (i = 0; i < frame->context->cap->method_comp.method_count; i++) {
	    if (frame->context->cap->method_comp.methods[i].cap_file_offset == method_offset) {
	      // XXXXX check for right method type
	      jardin_jcvm_jumpstart(frame->context, frame, &frame->context->cap->method_comp.methods[i]);
	      break;
	    }
	  }
	  if (i == frame->context->cap->method_comp.method_count) {
	    fprintf(stderr, "Error: no method_info found with offset 0x%.04x.\n", method_offset);
	    exit(1);
	  }
	}
	frame->pc += 3;
      }
      break;
    case OPCODE_invokeinterface:
      nyi(method_info, frame);
    break;
      case OPCODE_new:
      DEBUG_TWO_BYTE;
      {
	u2 class_index;
	class_index = TWO_BYTE_VALUE;
	jardin_jcvm_push_stack(frame, create_new_object(frame, class_index));
	printf("  put objectref 0x%.04x on frame stack\n", jardin_jcvm_get_stack(frame, -1));
	frame->pc += 3;
      }
      break;
    case OPCODE_newarray:
      DEBUG_ONE_BYTE;
      {
	unsigned char atype = ONE_BYTE_VALUE;
	jardin_jcvm_short count = jardin_jcvm_pop_stack(frame);
	jardin_jcvm_push_stack(frame, alloc_array(atype, count));
	frame->pc += 2;
      }
      break;
    case OPCODE_anewarray:
      nyi(method_info, frame);
      break;
    case OPCODE_arraylength:
      DEBUG_NULL_BYTE;
      {
	jardin_jcvm_reference arrayref;
	jardin_jcvm_short value;
	arrayref = jardin_jcvm_pop_stack(frame);
	if (arrayref == 0) {
	  fprintf(stderr, "Error: throw NullPointerException not yet implemented.\n");
	  exit(1);
	}
	if (object_heap[arrayref].type != ARRAY) {
	  fprintf(stderr, "Error: object_heap[%d].type != ARRAY\n", arrayref);
	  exit(1);
	}
	value = object_heap[arrayref].internal_data.array.count;
	jardin_jcvm_push_stack(frame, value);
	frame->pc += 1;
      }
      break;
    case OPCODE_athrow:
    case OPCODE_checkcast:
    case OPCODE_instanceof:
    case OPCODE_sinc_w:
    case OPCODE_iinc_w:
    case OPCODE_ifeq_w:
    case OPCODE_ifne_w:
    case OPCODE_iflt_w:
    case OPCODE_ifge_w:
    case OPCODE_ifgt_w:
    case OPCODE_ifle_w:
    case OPCODE_ifnull_w:
    case OPCODE_ifnonull_w:
    case OPCODE_if_acmpeq_w:
    case OPCODE_if_acmpne_w:
    case OPCODE_if_scmpeq_w:
    case OPCODE_if_scmpne_w:
    case OPCODE_if_scmplt_w:
    case OPCODE_if_scmpge_w:
    case OPCODE_if_scmpgt_w:
    case OPCODE_if_scmple_w:
    case OPCODE_goto_w:
    case OPCODE_getfield_a_w:
    case OPCODE_getfield_b_w:
    case OPCODE_getfield_s_w:
    case OPCODE_getfield_i_w:
      nyi(method_info, frame);
      break;
    case OPCODE_getfield_a_this:
      DEBUG_ONE_BYTE;
      {
	struct instance_fieldref instance_fieldref;
	int constant_pool_index = ONE_BYTE_VALUE;
	jardin_jcvm_reference this_ref = jardin_jcvm_get_local_variable(frame, 0);
	if (frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag != CONSTANT_InstanceFieldref) {
	  fprintf(stderr, "Error: cp_info.tag != CONSTANT_InstanceFieldref.\n");
	  exit(1);
	}
	if (this_ref == 0) {
	  fprintf(stderr, "Error: this = null (NullPointerException not yet implemented.\n");
	  exit(1);
	}
	instance_fieldref.tag = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].tag;
	instance_fieldref.class.internal_class_ref = (frame->context->cap->constant_pool_comp. constant_pool[constant_pool_index].info[0] << 8 ) | frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[1];
	instance_fieldref.token = frame->context->cap->constant_pool_comp.constant_pool[constant_pool_index].info[2];
	if (instance_fieldref.class.internal_class_ref > 32767) {
	  fprintf(stderr, "Eroor: instance_fieldref.class.external_class_ref not yet implemented.\n");
	} else {
	  jardin_jcvm_reference value = object_heap[this_ref].data[instance_fieldref.token - frame->context->cap->class_comp.classes[instance_fieldref.class.internal_class_ref].first_reference_token];
	  jardin_jcvm_push_stack(frame, value);
	    
	}
	frame->pc += 2;
      }
      break;
    case OPCODE_getfield_b_this:
    case OPCODE_getfield_s_this:
    case OPCODE_getfield_i_this:
    case OPCODE_putfield_a_w:
    case OPCODE_putfield_b_w:
    case OPCODE_putfield_s_w:
    case OPCODE_putfield_i_w:
    case OPCODE_putfield_a_this:
    case OPCODE_putfield_b_this:
    case OPCODE_putfield_s_this:
    case OPCODE_putfield_i_this:
    case OPCODE_impdep1:
    case OPCODE_impdep2:
      nyi(method_info, frame);
      break;
    default:
      fprintf(stderr, "Error: unknown opcode.\n");
      return;
    }
  }
  
  jardin_jcvm_release_frame(frame);
}

void jardin_jcvm_run(int applet_index) {
  struct jardin_jcvm_frame *null_frame;
  struct method_info *install = get_method_info_from_method_offset(jardin_jcvm_applets[applet_index].cap, jardin_jcvm_applets[applet_index].cap->applet_comp.applets[0].install_method_offset);
  
  null_frame = jardin_jcvm_allocate_frame(0, 3, &jardin_jcvm_applets[applet_index]);
  
  /* public static void install(byte[] bArray, short bOffset, byte bLength) */
  jardin_jcvm_push_stack(null_frame, 0);
  jardin_jcvm_push_stack(null_frame, 0);
  jardin_jcvm_push_stack(null_frame, 0);

  JARDIN_DEBUG("run HelloWorld.install()");

  jardin_jcvm_jumpstart(null_frame->context, null_frame, install);

  JARDIN_DEBUG("returned from HelloWorld.install()");

  jardin_jcvm_release_frame(null_frame);

  null_frame = jardin_jcvm_allocate_frame(0,0, &jardin_jcvm_applets[3]);
  install = get_method_info_from_method_offset(jardin_jcvm_applets[3].cap, 0x0008);
  JARDIN_DEBUG("run jardin.Jardin.run()");
  jardin_jcvm_jumpstart(null_frame->context, null_frame, install);
  JARDIN_DEBUG("returned from jardin.Jardin.run()");
  jardin_jcvm_release_frame(null_frame);
}
